// Copyright 2024 Esri
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//   https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

import SwiftUI
import ArcGIS

private class Model: ObservableObject {

    // The analysis overlay to be added to the scene.
    let analysisOverlay = AnalysisOverlay()

    let measurement: LocationDistanceMeasurement = {
        let point = Point(x: -119.5968, y: 37.7567, z: 58.5011, spatialReference: .wgs84)
        let measurement = LocationDistanceMeasurement(startLocation: point, endLocation: point)
        measurement.isVisible = false
        measurement.unitSystem = .metric
        return measurement
    }()

    init() {
        analysisOverlay.addAnalysis(measurement)
    }

    func moveDistanceMeasurement(point: Point) {
        if measurement.startLocation == measurement.endLocation {
            measurement.startLocation = point
        } else {
            measurement.endLocation = point
        }
        measurement.isVisible = true
    }

    func clearMeasurement() {
        measurement.isVisible = false
        let point = Point(x: 0, y: 0, z: 0, m: 0, spatialReference: .wgs84)
        measurement.startLocation = point
        measurement.endLocation = point
    }

}

struct ContentView: View {

    // An ObservableObject containing the scene, and analysis overlay.
    @StateObject private var model = Model()

    // The Yosemite Valley hotspots scene.
    @State private var scene: ArcGIS.Scene = {

        let portalItem = PortalItem(
            portal: .arcGISOnline(connection: .anonymous),
            id: Item.ID("7558ee942b2547019f66885c44d4f0b1")!
        )

        return Scene(item: portalItem)
    }()

    @State private var directDistanceText = "--"
    @State private var horizontalDistanceText = "--"
    @State private var verticalDistanceText = "--"

    var body: some View {
        VStack {            

            SceneView(scene: scene, analysisOverlays: [model.analysisOverlay])

                .onSingleTapGesture { _, scenePoint in
                    guard let scenePoint else { return }
                    model.moveDistanceMeasurement(point: scenePoint)
                }
                .onLongPressGesture { _, scenePoint in
                    guard let scenePoint else { return }
                    model.moveDistanceMeasurement(point: scenePoint)
                }

                .task {
                    for await measurements in model.measurement.measurements {
                        directDistanceText = measurements.directDistance.formatted(.distance)
                        horizontalDistanceText = measurements.horizontalDistance.formatted(.distance)
                        verticalDistanceText = measurements.verticalDistance.formatted(.distance)
                    }
                }

            Text("Direct: \(directDistanceText)")
            Text("Horizontal: \(horizontalDistanceText)")
            Text("Vertical: \(verticalDistanceText)")
            Spacer()

            .toolbar {
                ToolbarItemGroup(placement: .bottomBar) {
                    Button("Clear") {
                        // Resets the distance measurement.
                        model.clearMeasurement()
                    }
                }
            }

        }
    }

}

private extension FormatStyle where Self == Measurement<UnitLength>.FormatStyle {
    /// The format style for the distances.
    static var distance: Self {
        .measurement(width: .abbreviated, usage: .asProvided, numberFormatStyle: .number.precision(.fractionLength(2)))
    }
}


